<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\PostsExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\Posts\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\PostsImport;
use App\Models\Post;
use App\Models\TherapistCommunity;
use App\Models\BlogCategory;
use App\Models\Tag;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;

class TherapistCommunityPostsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist_community.add_blog');
        $this->middleware('permission:therapist_community.add_blog', ['only' => ['store']]);
        $this->middleware('permission:therapist_community.add_blog', ['only' => ['update']]);
        $this->middleware('permission:therapist_community.add_blog', ['only' => ['destroy']]);
        $this->middleware('permission:therapist_community.add_blog', ['only' => ['export']]);
        $this->middleware('permission:therapist_community.add_blog', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null, $therapist_community)
    {
        if ($req != null) {
            $therapist_community_posts =  $therapist_community->therapist_community_posts();
            if ($req->trash && $req->trash == 'with') {
                $therapist_community_posts =  $therapist_community_posts->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_community_posts =  $therapist_community_posts->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_community_posts = $therapist_community_posts->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_community_posts = $therapist_community_posts->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_community_posts = $therapist_community_posts->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_community_posts = $therapist_community_posts->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_community_posts = $therapist_community_posts->get();
                return $therapist_community_posts;
            }
            $therapist_community_posts = $therapist_community_posts->get();
            return $therapist_community_posts;
        }
        $therapist_community_posts = $therapist_community->therapist_community_posts()->withAll()->orderBy('id', 'desc')->get();
        return $therapist_community_posts;
    }


    /*********View All Posts  ***********/
    public function index(Request $request, TherapistCommunity $therapist_community)
    {
        $therapist_community_posts = $this->getter($request, null, $therapist_community);
        return view('super_admins.therapist_communitys.therapist_community_posts.index', compact('therapist_community_posts', 'therapist_community'));
    }

    /*********View Create Form of Post  ***********/
    public function create(TherapistCommunity $therapist_community)
    {
        $blog_categories = BlogCategory::active()->get();
        $tags = Tag::active()->get();

        return view('super_admins.therapist_communitys.therapist_community_posts.create', compact('blog_categories', 'therapist_community', 'tags'));
    }

    /*********Store Post  ***********/
    public function store(CreateRequest $request, TherapistCommunity $therapist_community)
    {

        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $request->merge(['created_by_user_id' => auth()->user()->id]);
            $data = $request->all();
            $data['image'] = uploadCroppedFile($request, 'image', 'therapist_community_posts');
            $therapist_community_post = $therapist_community->therapist_community_posts()->create($data);
            $therapist_community_post->slug = Str::slug($therapist_community_post->name . ' ' . $therapist_community_post->id, '-');
            $therapist_community_post->save();
            $therapist_community_post = $therapist_community->therapist_community_posts()->withAll()->find($therapist_community_post->id);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_community_posts.index', $therapist_community->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_community_posts.index', $therapist_community->id)->with('message', 'Post Created Successfully')->with('message_type', 'success');
    }

    /*********View Post  ***********/
    public function show(TherapistCommunity $therapist_community, Post $therapist_community_post)
    {
        if ($therapist_community->id != $therapist_community_post->therapist_community_id) {
            return redirect()->back()->with('message', 'TherapistCommunityEducation Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapist_communitys.therapist_community_posts.show', compact('therapist_community_post', 'therapist_community'));
    }

    /*********View Edit Form of Post  ***********/
    public function edit(TherapistCommunity $therapist_community, Post $therapist_community_post)
    {
        if ($therapist_community->id != $therapist_community_post->therapist_community_id) {
            return redirect()->back()->with('message', 'TherapistCommunityEducation Not Found')->with('message_type', 'error');
        }
        $blog_categories = BlogCategory::active()->get();
        $tags = Tag::active()->get();
        return view('super_admins.therapist_communitys.therapist_community_posts.edit', compact('therapist_community_post', 'blog_categories', 'therapist_community', 'tags'));
    }

    /*********Update Post  ***********/
    public function update(CreateRequest $request, TherapistCommunity $therapist_community, Post $therapist_community_post)
    {
        if ($therapist_community->id != $therapist_community_post->therapist_community_id) {
            return redirect()->back()->with('message', 'TherapistCommunityEducation Not Found')->with('message_type', 'error');
        }
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $request->merge(['last_updated_by_user_id' => auth()->user()->id]);
            $data = $request->all();
            if ($request->image) {
                $data['image'] = uploadCroppedFile($request, 'image', 'therapist_community_posts', $therapist_community_post->image);
            } else {
                $data['image'] = $therapist_community_post->image;
            }
            $therapist_community_post->update($data);
            $therapist_community_post = Post::find($therapist_community_post->id);
            $slug = Str::slug($therapist_community_post->name . ' ' . $therapist_community_post->id, '-');
            $therapist_community_post->update([
                'slug' => $slug
            ]);
            $therapist_community_post->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_community_posts.index', $therapist_community->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_community_posts.index', $therapist_community->id)->with('message', 'Post Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_community_posts = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_community_posts." . $extension;
        return Excel::download(new PostsExport($therapist_community_posts), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new PostsImport, $file);
        return redirect()->back()->with('message', 'Post Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE Post ***********/
    public function destroy(TherapistCommunity $therapist_community, Post $therapist_community_post)
    {
        if ($therapist_community->id != $therapist_community_post->therapist_community_id) {
            return redirect()->back()->with('message', 'TherapistCommunityEducation Not Found')->with('message_type', 'error');
        }
        $therapist_community_post->delete();
        return redirect()->back()->with('message', 'Post Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE Post ***********/
    public function destroyPermanently(Request $request, TherapistCommunity $therapist_community, $therapist_community_post)
    {
        if ($therapist_community->id != $therapist_community_post->therapist_community_id) {
            return redirect()->back()->with('message', 'TherapistCommunityEducation Not Found')->with('message_type', 'error');
        }
        $therapist_community_post = Post::withTrashed()->find($therapist_community_post);
        if ($therapist_community_post) {
            if ($therapist_community_post->trashed()) {
                if ($therapist_community_post->image && file_exists(public_path($therapist_community_post->image))) {
                    unlink(public_path($therapist_community_post->image));
                }
                $therapist_community_post->forceDelete();
                return redirect()->back()->with('message', 'Post Category Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Post Category is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Post Category Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore Post***********/
    public function restore(Request $request, TherapistCommunity $therapist_community, $therapist_community_post)
    {
        if ($therapist_community->id != $therapist_community_post->therapist_community_id) {
            return redirect()->back()->with('message', 'TherapistCommunityEducation Not Found')->with('message_type', 'error');
        }
        $therapist_community_post = Post::withTrashed()->find($therapist_community_post);
        if ($therapist_community_post->trashed()) {
            $therapist_community_post->restore();
            return redirect()->back()->with('message', 'Post Category Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Post Category Not Found')->with('message_type', 'error');
        }
    }
}
